import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import { notFound } from "next/navigation";
import { buildGuessPaperDetailMeta } from "@/lib/seo/guessPaperMeta";
import GuessPaperDetailClient from "./GuessPaperDetailClient";

export const dynamic = "force-dynamic";

async function fetchGuessPaperData(courseCode: string, paperCode: string) {
    try {
        const [res1, res2, res3, res4] = await Promise.all([
            API_ENDPOINTS.GUESS_PAPERS.DETAIL(courseCode, paperCode),
            API_ENDPOINTS.SUBSECTIONS(3, 35),
            API_ENDPOINTS.SUBSECTIONS(3, 36),
            API_ENDPOINTS.SUBSECTIONS(3, 43),
        ]);

        if (!res1.ok) return null;

        const [data1, data2, data3, data4] = await Promise.all([
            res1.json(),
            res2.json(),
            res3.json(),
            res4.json(),
        ]);

        return {
            guess_paper: data1.guess_paper,
            firstSection: data2 || null,
            secondSection: data3 || null,
            thirdSection: data4 || null
        };
    } catch {
        return null;
    }
}

export async function generateMetadata(
    { params }: { params: Promise<{ slug: string; code: string }> }
): Promise<Metadata> {
    const { slug, code } = await params;
    const data = await fetchGuessPaperData(slug, code);

    if (!data || !data.guess_paper) {
        return {
            title: "Guess Paper Detail",
        };
    }

    return buildGuessPaperDetailMeta(data.guess_paper, slug, code);
}

export default async function GuessPaperDetailPage({
    params
}: {
    params: Promise<{ slug: string; code: string }>;
}) {
    const { slug, code } = await params;

    // Decode slug and code manually if needed, or rely on Next.js param decoding
    // Usually params are already decoded, but safe to decode just in case if URLs are encoded
    // const courseCode = decodeURIComponent(slug);
    // const paperCode = decodeURIComponent(code);

    // Using raw params as they often come decoded in latest Next.js, but your previous code used decoded.
    // Let's stick to using them as passed to API, assuming API handles it or they are correct.

    const data = await fetchGuessPaperData(slug, code);

    if (!data || !data.guess_paper) {
        notFound();
    }

    return (
        <GuessPaperDetailClient
            courseCode={slug}
            guessPaperCode={code}
            productData={data.guess_paper}
            firstSection={data.firstSection}
            secondSection={data.secondSection}
            thirdSection={data.thirdSection}
        />
    );
}